package kap08_2;

import java.io.IOException;
import java.net.UnknownHostException;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

// Dieses Programm laesst sich nur mit den Libraries aus dem Prosys-OPC-SDK uebersetzen
// Eine Demoversion des SDK kann bei Prosys angefordert werden: https://prosysopc.com/products/opc-ua-java-sdk/
// Die Libraries muessen dann entsprechend der Anleitung in das Projekt/den Buildpath eingefuegt werden
import com.prosysopc.ua.ApplicationIdentity;
import com.prosysopc.ua.SecureIdentityException;
import com.prosysopc.ua.ServiceException;
import com.prosysopc.ua.UserIdentity;
import com.prosysopc.ua.client.ConnectException;
import com.prosysopc.ua.client.InvalidServerEndpointException;
import com.prosysopc.ua.client.UaClient;
import com.prosysopc.ua.stack.builtintypes.DataValue;
import com.prosysopc.ua.stack.builtintypes.LocalizedText;
import com.prosysopc.ua.stack.builtintypes.NodeId;
import com.prosysopc.ua.stack.builtintypes.QualifiedName;
import com.prosysopc.ua.stack.core.ApplicationDescription;
import com.prosysopc.ua.stack.core.ApplicationType;
import com.prosysopc.ua.stack.core.Attributes;
import com.prosysopc.ua.stack.core.Identifiers;
import com.prosysopc.ua.stack.core.ReferenceDescription;
import com.prosysopc.ua.stack.transport.security.SecurityMode;

public class OPCDemoClient {

	protected static String APP_NAME = "OPCDemoClient";
	protected static String APP_URI = "urn:localhost:UA:" + APP_NAME;
	protected static String PROD_URI = "urn:hof-university.de:NWT3_0:" + APP_NAME;

	protected UaClient client;

	public OPCDemoClient(String serverURI) throws SecureIdentityException, IOException, UnknownHostException, InvalidServerEndpointException, ConnectException, ServiceException {

		client = new UaClient(serverURI);//"opc.tcp://localhost:52520/OPCUA/SampleConsoleServer");

		// Beschreibung der Anwendung; diese wird an den Server geschickt
		ApplicationDescription appDescription = new ApplicationDescription();
		appDescription.setApplicationName(new LocalizedText(APP_NAME, Locale.ENGLISH));
		// 'localhost' (all lower case) in the URI is converted to the actual
		// host name of the computer in which the application is run
		appDescription.setApplicationUri(APP_NAME);
		appDescription.setProductUri(PROD_URI);
		appDescription.setApplicationType(ApplicationType.Client);

		final ApplicationIdentity identity = new ApplicationIdentity();
		identity.setApplicationDescription(appDescription);

		client.setApplicationIdentity(identity);

		// Security-Einstellungen; hier nur Username + Passwort
		client.setSecurityMode(SecurityMode.NONE);
		UserIdentity userIdentity = new UserIdentity("user1", "password");
		client.setUserIdentity(userIdentity);

		// Client mit Server verbinden
		client.connect();
		if (client.isConnected())
			System.out.format("Client verbunden mit %s\n", serverURI);
		else
			System.out.println("Client not connected");

	}

	public NodeId recursiveSearch(NodeId nodeId, String nodeName, String decoration) throws Exception {
		List<ReferenceDescription> references;
		NodeId myNodeId = null;
		String[] terms = nodeName.split(":");

		System.out.format("\t%sRekursive Suche: %s\n",decoration,terms[0]);

		client.getAddressSpace().setMaxReferencesPerNode(1000);

		references = client.getAddressSpace().browse(nodeId);
		for (Iterator<ReferenceDescription> iterator = references.iterator(); iterator.hasNext();)
		{
			// alle Nodes in der naechsten Ebene pruefen, ob sie zum gesuchten term passen
			ReferenceDescription ref = iterator.next();
			QualifiedName name = ref.getBrowseName();
			if(name.getName().equals(terms[0])) {
				// wenn der erste Suchterm gleich dem Namen ist
				System.out.format("\t  %sgefunden: %s ", decoration,terms[0]);
				System.out.format("%s \n",ref);
				if (terms.length == 1) {
					// und wir nur einen term (uebrig) hatten, dann fertig
					return client.getAddressSpace().getNamespaceTable().toNodeId(ref.getNodeId());
				}
				else {
					// wenn noch Terms uebrig sind, neuen um einen term kuerzeren nodeName bauen
					String search ="";
					for(int i=1; i < terms.length;i++) {
						search = search + terms[i] + ":";
					}
					search = search.substring(0,search.length()-1);
					// und rekursiv weitersuchen
					return  recursiveSearch(
							client.getAddressSpace().getNamespaceTable().toNodeId(ref.getNodeId()),
							search,
							decoration + "  " // jede Zeile weiter einruecken 
							);
				}
			}
		}
		// wenn das Programm hier landet, hat es nichts gefunden, also Programm beenden
		throw new Exception("Rekursive Suche gescheitert ");
	}

	public void readNode(String browseName) throws Exception {
		System.out.format("==========================================\nreadNode %s\n", browseName);

		NodeId nodeId = Identifiers.RootFolder;
		DataValue name = client.readAttribute(nodeId, Attributes.BrowseName);

		System.out.format("  Durchsuche Adressbaum ausgehend von %s\n",name);

		nodeId = recursiveSearch(nodeId, browseName, "");
		name = client.readAttribute(nodeId, Attributes.BrowseName);

		System.out.format("  gefunden: %s\n",name);

		System.out.println("\n  Lese den Wert; 10 Aufrufe alle zwei Sekunden");
		for(int i = 0; i< 10; i++) {
			DataValue value = client.readAttribute(nodeId, Attributes.Value);
			System.out.format("\tWert %1.10f",value.getValue().toNumber().doubleValue());
			System.out.format(" aktualisiert %s\n",value.getServerTimestamp());

			Thread.sleep(2*1000);
		}

		System.out.format("readNode Ende\n==========================================\n");		
	}

	public void disconnect() {
		System.out.println("Verbindung abbauen");
		client.disconnect();
	}

	public static void main(String[] args) throws Exception {
		String server = "opc.tcp://milo.digitalpetri.com:62541/milo";

		System.out.format("%s startet und verbindet zu %s\n",APP_NAME, server);

		OPCDemoClient myClient = new OPCDemoClient(server);

		myClient.readNode("Objects:Dynamic:RandomDouble");

		myClient.disconnect();

		System.out.println("\nThats all, folks...");
	}
}
